<?php
/**
 * Contact Field
 *
 * @package     AutomatorWP\Integrations\iMember360\Triggers\Contact_Field
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_iMember360_Contact_Field extends AutomatorWP_Integration_Trigger {

    public $integration = 'imember360';
    public $trigger = 'imember360_contact_field';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User contact has a field greater than, less than, contains, or equal to a specific value', 'automatorwp-pro' ),
            'select_option'     => __( 'User contact has a field <strong>greater than, less than, contains or equal to</strong> a specific value', 'automatorwp-pro' ),
            /* translators: %1$s: Field. %2$s: Condition. %3$s: Value. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'User contact has the field %1$s %2$s %3$s %4$s time(s)', 'automatorwp-pro' ), '{field}', '{condition}', '{value}', '{times}' ),
            /* translators: %1$s: Field. %2$s: Condition. %3$s: Value. */
            'log_label'         => sprintf( __( 'User contact has the field %1$s %2$s %3$s', 'automatorwp-pro' ), '{field}', '{condition}', '{value}' ),
            'action'            => 'init',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'field' => array(
                    'from' => 'field',
                    'default' => __( 'field', 'automatorwp-pro' ),
                    'fields' => array(
                        'field' => array(
                            'name' => __( 'Field:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'classes' => 'automatorwp-selector',
                            'options_cb' => array( $this, 'fields_options_cb' ),
                            'default' => ''
                        )
                    )
                ),
                'condition' => array(
                    'from' => 'condition',
                    'fields' => array(
                        'condition' => array(
                            'name' => __( 'Condition:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'equal'                     => __( 'equal to', 'automatorwp-pro' ),
                                'not_equal'                 => __( 'not equal to', 'automatorwp-pro' ),
                                'greater_than'              => __( 'greater than', 'automatorwp-pro' ),
                                'less_than'                 => __( 'less than', 'automatorwp-pro' ),
                                'greater_or_equal'          => __( 'greater or equal to', 'automatorwp-pro' ),
                                'less_or_equal'             => __( 'less or equal to', 'automatorwp-pro' ),
                                'begins_with'               => __( 'begins with', 'automatorwp-pro' ),
                                'ends_with'                 => __( 'ends with', 'automatorwp-pro' ),
                                'contains'                  => __( 'contains', 'automatorwp-pro' ),
                                'not_contains'              => __( 'not contains', 'automatorwp-pro' )
                            ),
                            'default' => 'equal'
                        )
                    )
                ),
                'value' => array(
                    'from' => 'value',
                    'default' => __( 'value', 'automatorwp-pro' ),
                    'fields' => array(
                        'value' => array(
                            'name' => __( 'Value:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Get fields options
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function fields_options_cb() {

        global $i4w;

        $options = array();

        // Get site contact fields
        if ( ! empty( $i4w->dbsFIELDS ) ) {
            foreach ( $i4w->dbsFIELDS AS $field_name => $field_label ) {
                $options[$field_name] = $field_label;
            }
        }

        return $options;

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     */
    public function listener() {

        global $SESSION;

        // Bail if in admin area
        if( is_admin() ) {
            return;
        }

        $user_id = get_current_user_id();

        // Bail if user is not logged in
        if( $user_id === 0 ) {
            return;
        }

        if ( I4WADMIN ) {
            return NULL;
        }

        $remote_logged_in = $SESSION['DBSHORT']['id'];

        // Bail if user account is not remote logged in
        if ( empty( $remote_logged_in ) ) {
            return;
        }

        // Trigger the user is subscribed to a list
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        global $SESSION;

        // Shorthand
        $field = $trigger_options['field'];
        $condition = $trigger_options['condition'];
        $value = $trigger_options['value'];

        $field_value = $SESSION['DBSHORT'][ strtolower( $field ) ];

        switch ( $condition ) {
            case 'equal':
                $match = ( $field_value == $value );
                break;
            case 'not_equal':
                $match = ( $field_value != $value );
                break;
            case 'less_than':
                $match = ( $field_value <  $value );
                break;
            case 'greater_than':
                $match = ( $field_value >  $value );
                break;
            case 'greater_or_equal':
                $match = ( $field_value >= $value );
                break;
            case 'less_or_equal':
                $match = ( $field_value <= $value );
                break;
            case 'begins_with':
                $match = ( substr( $field_value,0, strlen( $value ) ) == $value );
                break;
            case 'ends_with':
                $match = ( substr( $field_value,strlen( $value ) * -1 ) == $value);
                break;
            case 'contains':
                $match = ( FALSE !== stripos( $field_value, $value ) );
                break;
            case 'not_contains':
                $match = ( FALSE === stripos( $field_value, $value ) );
                break;
            default:
                $match = false;
                break;
        }

        // Don't deserve if user field doesn't match with the trigger options
        if( ! $match ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_iMember360_Contact_Field();